#!/usr/bin/env python
#
# Copyright 2004,2005,2006,2007 Free Software Foundation, Inc.
#
# This file is part of GNU Radio
#
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 51 Franklin Street,
# Boston, MA 02110-1301, USA.
#

# print "Loading revised usrp_oscope with additional options for scopesink..."

from gnuradio import gr, gru, air
from gnuradio import usrp
from gnuradio import eng_notation
from gnuradio.eng_option import eng_option
from gnuradio.wxgui import stdgui2, scopesink2, form, slider
from optparse import OptionParser
import wx
import sys

risetime_threshold_db = 48.0   # The minimum change for pulse leading edge in dB per bit time (Assume value for 8 MHz BW)
data_rate = 1000000.0        # Data rate in bits per second
chip_rate = data_rate*2.0     # Two chips to a bit so rate is double

def pick_subdevice(u):
    """
    The user didn't specify a subdevice on the command line.
    If there's a daughterboard on A, select A.
    If there's a daughterboard on B, select B.
    Otherwise, select A.
    """
    if u.db(0,0).dbid() >= 0:       # dbid is < 0 if there's no d'board or a problem
        return (0, 0)
    if u.db(1,0).dbid() >= 0:
        return (1, 0)
    return (0, 0)


class app_flow_graph(stdgui2.std_top_block):
    def __init__(self, frame, panel, vbox, argv):
        stdgui2.std_top_block.__init__(self, frame, panel, vbox, argv)

        self.frame = frame
        self.panel = panel

        parser = OptionParser(option_class=eng_option)
        parser.add_option("-R", "--rx-subdev-spec", type="subdev", default=None,
                          help="select USRP Rx side A or B (default=first one with a daughterboard)")
        parser.add_option("-d", "--decim", type="int", default=8,
                          help="set fgpa decimation rate to DECIM [default=%default]")
    	parser.add_option("-f", "--freq", type="eng_float", default=1090.0,
                          help="set receive frequency to MHz [default=%default]", metavar="FREQ")
        parser.add_option("-g", "--gain", type="eng_float", default=None,
                          help="set gain in dB (default is midpoint)")
        parser.add_option("-8", "--width-8", action="store_true", default=False,
                          help="Enable 8-bit samples across USB")
        parser.add_option("-n", "--frame-decim", type="int", default=1,
                          help="set oscope frame decimation factor to n [default=1]")
        parser.add_option("-v", "--v-scale", type="eng_float", default=200,
                          help="set oscope initial V/div to SCALE [default=%default]")
        parser.add_option("-t", "--t-scale", type="eng_float", default=24e-6,
                          help="set oscope initial s/div to SCALE [default=25us]")
    	parser.add_option("-T", "--thresh", type="int", default=10,
                          help="set valid pulse threshold to THRESH [default=%default]")
        (options, args) = parser.parse_args()
        if len(args) != 0:
            parser.print_help()
            sys.exit(1)

        self.show_debug_info = True

	options.freq *= 1e6

        # build the graph

	if options.decim < 8:
		self.fpga_filename="std_4rx_0tx.rbf" #Min decimation of this firmware is 4. contains 4 Rx paths without halfbands and 0 tx paths.
	       	self.u = usrp.source_c(decim_rate=options.decim, fpga_filename=self.fpga_filename)
	else :
                #standard fpga firmware "std_2rxhb_2tx.rbf" contains 2 Rx paths with halfband filters and 2 tx paths (the default) min decimation 8
        	self.u = usrp.source_c(decim_rate=options.decim)
        if options.rx_subdev_spec is None:
            options.rx_subdev_spec = pick_subdevice(self.u)
        self.u.set_mux(usrp.determine_rx_mux_value(self.u, options.rx_subdev_spec))

        if options.width_8:
            width = 8
            shift = 8
            format = self.u.make_format(width, shift)
            #print "format =", hex(format)
            r = self.u.set_format(format)
            #print "set_format =", r

        # determine the daughterboard subdevice we're using
        self.subdev = usrp.selected_subdev(self.u, options.rx_subdev_spec)
	#self.subdev.set_bw(5e6)  # only valid for dbsrx

	chan_rate = 8000000
        input_rate = self.u.adc_freq() / self.u.decim_rate()
	if input_rate < chan_rate:
            raise ValueError, "Invalid input rate %d. Must be 8000000 sps or higher" % (input_rate)
	if input_rate >= 10000000:
		chan_rate = 10000000    # Higher Performance Receiver
	# if rate is not supported then resample
	if input_rate != chan_rate:
		raise Error, "Resampling Unsupported"
		#interp = gru.lcm(input_rate, chan_rate)/input_rate
		#decim  = gru.lcm(input_rate, chan_rate)/chan_rate
		#print interp, decim
		#self.resamp = blks.rational_resampler_ccf(self, interp, decim)

	# Calculate the leading edge threshold per sample time
	leading_edge = risetime_threshold_db/(chan_rate/data_rate)
	# The number of following samples above threshold needed to make a sample a valid pulse position
	valid_pulse_position = 2
	if chan_rate == 10000000:
		valid_pulse_position = 3
	self.mag = gr.complex_to_mag()
	self.detect = air.ms_pulse_detect(leading_edge, options.thresh, valid_pulse_position)
	self.sync = air.ms_preamble(chan_rate)
	self.frame = air.ms_framer(chan_rate)
	self.cvt = air.ms_cvt_float()
        self.scope = scopesink2.scope_sink_f(self, panel, sample_rate=chan_rate,
                                            frame_decim=options.frame_decim,
                                            v_scale=options.v_scale,
                                            t_scale=options.t_scale)
	#if chan_rate == 10000000:
        #self.connect(self.u, self.resamp, self.mag, self.detect)
	#else :
        self.connect(self.u, self.mag, self.detect)
	self.connect((self.detect, 0), (self.sync, 0))
	self.connect((self.detect, 1), (self.sync, 1))
	self.connect((self.sync, 0), (self.frame, 0))
	self.connect((self.sync, 1), (self.frame, 1))
        self.connect((self.frame, 0), (self.cvt, 0))
        self.connect((self.frame, 1), (self.cvt, 1))
        self.connect((self.cvt, 0), (self.scope, 0))
        self.connect((self.cvt, 1), (self.scope,1))
        self.connect((self.cvt, 2), (self.scope,2))

        self._build_gui(vbox)

        # set initial values

        if options.gain is None:
            # if no gain was specified, use the mid-point in dB
            g = self.subdev.gain_range()
            options.gain = float(g[0]+g[1])/2


        self.set_gain(options.gain)

        if self.show_debug_info:
            self.myform['decim'].set_value(self.u.decim_rate())
            self.myform['fs@usb'].set_value(self.u.adc_freq() / self.u.decim_rate())
            self.myform['dbname'].set_value(self.subdev.name())
            self.myform['baseband'].set_value(0)
            self.myform['ddc'].set_value(0)

        if not(self.set_freq(options.freq)):
            self._set_status_msg("Failed to set initial frequency")


    def _set_status_msg(self, msg):
        self.frame.GetStatusBar().SetStatusText(msg, 0)

    def _build_gui(self, vbox):

        def _form_set_freq(kv):
            return self.set_freq(kv['freq'])

        vbox.Add(self.scope.win, 10, wx.EXPAND)

        # add control area at the bottom
        self.myform = myform = form.form()
        hbox = wx.BoxSizer(wx.HORIZONTAL)
        hbox.Add((5,0), 0, 0)
        myform['freq'] = form.float_field(
            parent=self.panel, sizer=hbox, label="Center freq", weight=1,
            callback=myform.check_input_and_call(_form_set_freq, self._set_status_msg))

        hbox.Add((5,0), 0, 0)
        g = self.subdev.gain_range()
        myform['gain'] = form.slider_field(parent=self.panel, sizer=hbox, label="Gain",
                                           weight=3,
                                           min=int(g[0]), max=int(g[1]),
                                           callback=self.set_gain)

        hbox.Add((5,0), 0, 0)
        vbox.Add(hbox, 0, wx.EXPAND)

        self._build_subpanel(vbox)

    def _build_subpanel(self, vbox_arg):
        # build a secondary information panel (sometimes hidden)

        # FIXME figure out how to have this be a subpanel that is always
        # created, but has its visibility controlled by foo.Show(True/False)

        def _form_set_decim(kv):
            return self.set_decim(kv['decim'])

        if not(self.show_debug_info):
            return

        panel = self.panel
        vbox = vbox_arg
        myform = self.myform

        #panel = wx.Panel(self.panel, -1)
        #vbox = wx.BoxSizer(wx.VERTICAL)

        hbox = wx.BoxSizer(wx.HORIZONTAL)
        hbox.Add((5,0), 0)

        myform['decim'] = form.int_field(
            parent=panel, sizer=hbox, label="Decim",
            callback=myform.check_input_and_call(_form_set_decim, self._set_status_msg))

        hbox.Add((5,0), 1)
        myform['fs@usb'] = form.static_float_field(
            parent=panel, sizer=hbox, label="Fs@USB")

        hbox.Add((5,0), 1)
        myform['dbname'] = form.static_text_field(
            parent=panel, sizer=hbox)

        hbox.Add((5,0), 1)
        myform['baseband'] = form.static_float_field(
            parent=panel, sizer=hbox, label="Analog BB")

        hbox.Add((5,0), 1)
        myform['ddc'] = form.static_float_field(
            parent=panel, sizer=hbox, label="DDC")

        hbox.Add((5,0), 0)
        vbox.Add(hbox, 0, wx.EXPAND)


    def set_freq(self, target_freq):
        """
        Set the center frequency we're interested in.

        @param target_freq: frequency in Hz
        @rypte: bool

        Tuning is a two step process.  First we ask the front-end to
        tune as close to the desired frequency as it can.  Then we use
        the result of that operation and our target_frequency to
        determine the value for the digital down converter.
        """
        r = usrp.tune(self.u, 0, self.subdev, target_freq)

        if r:
            self.myform['freq'].set_value(target_freq)     # update displayed value
            if self.show_debug_info:
                self.myform['baseband'].set_value(r.baseband_freq)
                self.myform['ddc'].set_value(r.dxc_freq)
            return True

        return False

    def set_gain(self, gain):
        self.myform['gain'].set_value(gain)     # update displayed value
        self.subdev.set_gain(gain)

    def set_decim(self, decim):
	# For now disallow setting
	print "set_decim not allowed"
        #ok = self.u.set_decim_rate(decim)
        #if not ok:
        #    print "set_decim failed"
        input_rate = self.u.adc_freq() / self.u.decim_rate()
        self.scope.set_sample_rate(input_rate)
        if self.show_debug_info:  # update displayed values
            self.myform['decim'].set_value(self.u.decim_rate())
            self.myform['fs@usb'].set_value(self.u.adc_freq() / self.u.decim_rate())
        return ok

def main ():
    app = stdgui2.stdapp(app_flow_graph, "USRP Mode S O'scope", nstatus=1)
    app.MainLoop()

if __name__ == '__main__':
    main ()
