/*
 *    Copyright 2018 Yizheng Huang
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 */
package app.fedilab.android.mastodon.watermark.androidwm;

import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.widget.ImageView;

import app.fedilab.android.mastodon.watermark.androidwm.listener.DetectFinishListener;
import app.fedilab.android.mastodon.watermark.androidwm.task.FDDetectionTask;
import app.fedilab.android.mastodon.watermark.androidwm.task.LSBDetectionTask;
import io.reactivex.annotations.NonNull;

/**
 * This is for detecting the invisible watermark in one picture.
 *
 * @author huangyz0918 (huangyz0918@gmail.com)
 */
public final class WatermarkDetector {
    private final Bitmap imageWithWatermark;
    private final boolean isLSB;

    private WatermarkDetector(
            @NonNull Bitmap imageWithWatermark,
            boolean isLSB) {
        this.imageWithWatermark = imageWithWatermark;
        this.isLSB = isLSB;
    }

    /**
     * to get an instance form class.
     *
     * @return instance of {@link WatermarkDetector}
     */
    public static WatermarkDetector create(@NonNull Bitmap imageWithWatermark, boolean isLSB) {
        return new WatermarkDetector(imageWithWatermark, isLSB);
    }

    /**
     * to get an instance form class.
     * If the imageView has no src or bitmap image, it will throws a {@link NullPointerException}.
     *
     * @return instance of {@link WatermarkDetector}
     */
    public static WatermarkDetector create(ImageView imageView, boolean isLSB) {
        BitmapDrawable drawable = (BitmapDrawable) imageView.getDrawable();
        return new WatermarkDetector(drawable.getBitmap(), isLSB);
    }

    /**
     * The method for watermark detecting.
     */
    public void detect(DetectFinishListener listener) {
        if (isLSB) {
            new LSBDetectionTask(listener).execute(imageWithWatermark);
        } else {
            new FDDetectionTask(listener).execute(imageWithWatermark);
        }
    }
}
